/****************************  Drm Decoder Library  **************************

                     (C) Copyright Fraunhofer IIS (2007-2013)
                               All Rights Reserved

   $Id$
   Author(s):   Christian Griebel
   Description: DRM common interface wrapper

   This software and/or program is protected by copyright law and international
   treaties. Any reproduction or distribution of this software and/or program,
   or any portion of it, may result in severe civil and criminal penalties, and
   will be prosecuted to the maximum extent possible under law.

******************************************************************************/

/*!
 * \file   drm_dec_lib.h
 * \brief  CDK DRM decoder library interface header file.
 *
\page INTRO Introduction

\section SCOPE Scope

This document describes the interface and usage of the
<CENTER>
<b>Digital Radio Mondiale (DRM30/DRM+) Decoder Library for

Fixed-point Processors - Core Design Kit</b>

Reference and template code for ETSI ES 201 980 v4.1.1 conform

DRM30/DRM+ decoders on fixed point processors
</CENTER>
developed by the <b>Fraunhofer Institute for Integrated Circuits IIS</b>.

\section SWLicense Software License

<HR>

If this document is part of a software and/or program delivery, then
this software and/or program delivery is subject to and covered by the
``Software License Agreement'' or ``Software Evaluation Agreement'' from
Fraunhofer IIS you should have already signed.

Please note: In case this software delivery contains source code, this
source code is deemed Confidential Information.

You may use this software and/or program only
under the terms and conditions described in the above mentioned agreement.

If you are in possession of this software and/or program, although you or
your company has not signed any of the agreements mentioned above, then you are
NOT ALLOWED to use the software and/or program. Any use, interference
with, disclosure or copying of the software and/or program is
unauthorized and prohibited. Please inform us immediately and destroy
this document. Thank you for your cooperation.

<HR>

This documentation itself is also covered by the above mentioned
agreements and subject to the same restrictions and limitations
described above.

\section DRMDecoderBasics DRM Decoder Basics

The Fraunhofer IIS CDK DRM30/DRM+ Decoder is a complete implementation of the HE-AAC&nbsp;v2 and xHE-AAC decoding algorithms specified in ETSI ES 201 980 version 4.1.1 and additional modules needed to decode bitstreams conform to ETSI ES 201 980.

\b Features:
- Input: ETSI ES 201 980 compliant data stream
- Output: 16 bit Linear PCM
- Channel configurations: mono, stereo
- Sampling frequencies: depend on used audio codec as defined in ETSI ES 201 980

The Fraunhofer IIS CDK DRM30/DRM+ Decoder Library offers an application programming interface
in C (function-style). The library is designed to be fully thread-safe and reentrant
with multiple decoder instances.

Please refer to the example applications for more information on how to use this CDK DRM30/DRM+ decoder.

\section DRMDecoderConcealment DRM Decoder Error Concealment

The decoder implements error concealment routines as defined by ETSI ES 201 980. Interpolation concealment is implemented for AAC coding. 
For xHE-AAC coding interpolation concealment is not mandated by the standard. In favor of a smaller memory footprint and less decoder signal delay an extrapolation concealment strategy is used here.

The concealment behavior may be configured by using the DRMDEC_FHG_SetParam() interface with the DRM_AAC_CONCEAL parameters. 
Note that these parameters mainly effect the AAC error concealment. Due to the more complex internal structure of xHE-AAC coding (combined speech and audio coding paths) the concealment routines run largely independent from these parameters in order to achieve the best concealment performance.

\page USAGE Usage

The xHE-AAC super frame transport format has some properties which need to be considered by the framework code.

For HE-AAC&nbsp;v2 there is a fixed amount (5 or 10 depending on the configuration) audio frames per DRM super frame.
This means that for each incoming super frame which is worth 400&nbsp;ms (200&nbsp;ms for DRM+) decoded output PCM samples can be obtained from the decoder which represent exactly the same amount of time.
If a super frame is lost or heavily corrupted it is straight forward for the decoder to know how many audio frames need to be concealed.

The DRM xHE-AAC transport mechanism basically decouples the assignment of audio frames from the fixed byte size and duration of a super frame (see section 5.3.1.3, ETSI ES 201 980).
This gives the encoder more degrees of freedom to exploit the bit reservoir which improves the audio quality especially at low bit rates.
The variable length directory section of a xHE-AAC audio super frame can signal the beginning of 0 up to 15 audio frames.
For a given audio configuration an average amount of audio frames per super frame can be calculated.
However this is typically not an integer number. For any given super frame the decoder may be able to decode between 0 and 16 audio frames.
To cope for this variability (difference between average number of audio frames and actual number of audio frames for a giver super frame) the following needs to be considered:
- To avoid that the decoder runs out of input data at any time some amount of initial bitstream buffering (hold-off) is required. This is typically in the range of 1-3 super frames and the required amount is calculated by the decoder automatically.
During this time the decoder returns with DRMDEC_NEED_DATA and no output samples or DRMDEC_OK and silence if the DRMDEC_FORCE_OUTPUT flag is used.
- During regular decoding operation up to a certain amount of time needs to remain buffered. This can be either done in the input bitstream domain (see \ref CallingSequenceRT) or in the output PCM domain (see \ref CallingSequenceFileBased)

If a DRM xHE-AAC super frame is lost or heavily corrupted the decoder might not be able to deduce the exact amount of lost audio frames.
Instead a minimum amount of concealed audio frames is inserted.

\section CallingSequence Calling Sequence

\subsection CallingSequenceRT Decode with DRM_DECODE_FORCE_OUTPUT flag

This is generally the recommended way of integrating the decoder into a DRM receiver application. See demo_main.cpp for details.

Prerequisites:
- A FIFO buffer for the input bitstream has to be provided.
- The decoder needs to be called in a time grid at which the decoded output samples are needed for play-out.
If this time grid is not inherently available it may be deduced from the duration of incoming super frames (400 or 200&nbsp;ms).

Advantages:
- xHE-AAC transport time variability is buffered in bitstream domain rather than in the output PCM domain which is the most memory efficient way.
- Potentially lost xHE-AAC audio frames are automatically filled with concealed frames in contrast to returning no samples and DRMDEC_NEED_DATA.

Calling Sequence:
-# Call DRMDEC_FHG_Open() to open and retrieve a handle to an opaque DRM decoder instance.
-# If received config has changed call DRMDEC_FHG_Config() with flag DRM_CONFIG_FULL_RESET and clear bitstream FIFO buffer. Otherwise do nothing.
-# Check for new DRM audio super frame from bitstream file or stream and add to client-supplied input buffer.
   This can be done by calling the function inputRead() in demo_main.cpp but can also be done by using any other function with the same functionality.
-# If new input data is available append it to the FIFO buffer.
-# Call DRMDEC_FHG_DecodeFrame() using the DRM_DECODE_FORCE_OUTPUT flag
   - The decoder will consume data from the FIFO buffer if needed.
   - The DRMDEC_INFO structure contains information on the decoded audio frame and is required to correctly initialize an audio output device
     (e.g.: number of channels, sampling frequency, frame size).
   - The settings for the audio output device might change from one audio frame to another and should always be checked for changes
   - Framework is required to repeatedly recall DRMDEC_FHG_DecodeFrame() until enough PCM data has been decoded.
-# As long as there is still input data to be decoded, goto 2.
-# Call DRMDEC_FHG_Close() to deallocate all DRM decoder structures.

\subsection CallingSequenceFileBased Decode until DRMDEC_NEED_DATA is signaled.

Prerequisites:
- The framework or sound device needs to be able to cope with missing decoded output data. E.g. by inserting silence.
- The time variability of the xHE-AAC transport scheme needs to be buffered in the output PCM domain.

Advantages:
- DRMDEC_FHG_DecodeFrame() can be simply called until it returns DRMDEC_NEED_DATA (all input bitstream data has been consumed).

Calling Sequence:
-# Call DRMDEC_FHG_Open() to open and retrieve a handle to an opaque DRM decoder instance.
-# Call DRMDEC_FHG_Config() to setup active configuration for next DRM audio super frame if new/changed
   - Do this whenever the active configuration for the next DRM audio super frame changes
   - Any configuration will be pending and processed with next DRM audio super frame
-# Check for new DRM audio super frame from bitstream file or stream and add to client-supplied input buffer.
   This can be done by calling the function inputRead() in demo_main.cpp but can also be done by using any other function with the same functionality.
-# Call DRMDEC_FHG_DecodeFrame() to fill the bitstream from the client supplied external bitstream input buffer into the DRM decoder's internal bitstream buffer.
   - If the decoder is given valid input data the function call returns the PCM coded audio samples of one decoded audio frame.
   - The DRMDEC_INFO structure contains information on the decoded audio frame and is required to correctly initialize an audio output device
     (e.g.: number of channels, sampling frequency, frame size).
   - The settings for the audio output device might change from one audio frame to another and should always be checked for changes
   - One DRM audio super frame may contain more than one audio frame. This function will accept new audio super frame data only after internal buffer for audio
     frames has been fully processed. Framework is required to repeatedly recall DRMDEC_FHG_DecodeFrame() until all internally stored audio frames have been decoded.
     After that, this function can accept the next audio super frame if already present in input buffer.
-# As long as there is still input data to be decoded, goto 2.
-# Call DRMDEC_FHG_Close() to deallocate all DRM decoder structures.

\page FunctionReference Function Reference

The CDK DRM Decoder Library offers a high-level interface for decoding of DRM bitstreams. This library is capable of handling mono and stereo signals. The following header files are provided for usage in C/C++ programs:

\b drm_dec_lib.h: DRM decoder library function prototypes.

The decoder core resides in statically linkable libraries called for example libDrmDec.a (UNIX) or CDK_DrmDec_Lib.lib (Microsoft Windows).

\b demo_main.cpp contains example source code demonstrating how to interface with the decoder core.

*/

#ifndef DRM_DECODER_LIB_H
#define DRM_DECODER_LIB_H

/*
 *  Settings:
 */
//#define DRM_WRKL_MEASUREMENT

/*
 *  Includes:
 */
#include "machine_type.h"

#include "CDK_audio.h"          /* CDK specific includes. */


/*
 *  Global constants:
 */
#define DRMDEC_MAX_NUM_IO_BUFFERS        (  2 )

/*!
 * This enum defines the available audio coding types.
 */
typedef enum
{
  AUDIO_CODING_AAC = 0,                               /*!< */
  AUDIO_CODING_CELP,                                  /*!< */
  AUDIO_CODING_HVXC,                                  /*!< */
  AUDIO_CODING_XHE                                    /*!< */

} AudioCodingType;

/*!
 * This enum defines the different audio mode types.
 */
typedef enum
{
  AUDIO_MODE_MONO = 0,                                /*!< */
  AUDIO_MODE_PARAMETRICSTEREO,                        /*!< */
  AUDIO_MODE_STEREO,                                  /*!< */
  AUDIO_MODE_RES                                      /*!< */

} AudioModeType;

/*!
 * DRM decoder lib errors.
 */
typedef enum
{
  DRMDEC_OK                            = 0x0000,      /*!< No error occurred. */
  DRMDEC_NEED_DATA,                                   /*!< The decoder will need more input data before any further decoding is possible. */

  DRMDEC_UNKNOWN                       = 0x0005,      /*!< Error condition is of unknown reason or from a third party module. */

  /* Drm decoder open and init errors */
  drm_dec_lib_open_error_start         = 0x0100,
  DRMDEC_OUT_OF_MEMORY,                               /*!< Could not get enough memory to create the DRM specific structures. */
  DRMDEC_INVALID_CONFIGURATION,                       /*!< The configuration is either wrong, not supported or has not been set yet. Any previously accepted configuration will be kept. */
  DRMDEC_INITIALIZATION_ERROR,                        /*!< Internal AAC decoder could not be initialized. New config is needed. */
  DRMDEC_INVALID_PARAMETER,                           /*!< The parameter to be set doesn't exist or is disabled. */
  DRMDEC_SET_PARAM_FAIL,                              /*!< Unable to set the given parameter. Most probably the value was out of range. */
  drm_dec_lib_open_error_end,

  /* Drm decoder decode frame errors */
  drm_dec_lib_decode_error_start       = 0x0200,
  DRMDEC_INVALID_HANDLE,                              /*!< The DRM decoder handle provided is not valid. */
  DRMDEC_INVALID_PCM_BUFFER_HANDLE,                   /*!< The handle to the PCM buffer is pointing on forbidden memory. */
  DRMDEC_INVALID_DECINFO_PTR,                         /*!< The pointer on the buffer receiving the decoder info pointer is invalid. */
  drm_dec_lib_decode_error_end,

  /* Drm parser errors */
  drm_dec_parser_error_start           = 0x0400,
  DRMDEC_PARSER_OPEN_ERROR,                           /*!< Unable to open the DRM super frame parser. */
  drm_dec_parser_error_end,

  /* AAC decoder errors */
  drm_dec_aac_error_start              = 0x0800,
  DRMDEC_AAC_OPEN_ERROR,                              /*!< Unable to open the HE-AAC&nbsp;v2 decoder. */
  DRMDEC_DECODE_FRAME_ERROR,                          /*!< An error occurred during decoding of one audio frame. */
  drm_dec_aac_error_end

} DRM_DECODER_ERROR;

#define IS_DRM_LIB_ERROR(err) (!!((err) >= drm_dec_lib_open_error_start) && ((err) <= drm_dec_lib_decode_error_end))
#define IS_PARSER_ERROR(err) (!!((err) >= drm_dec_parser_error_start) && ((err) <= drm_dec_parser_error_end))
#define IS_AAC_ERROR(err) (!!((err) >= drm_dec_aac_error_start) && ((err) <= drm_dec_aac_error_end))

/*!
 * This enumeration defines the available output modes.
 */
typedef enum
{
  DRM_OUTPUT_NORMAL   = 0,   /*!< Output the channel configuration found in the stream. No up-/downmixing applied.       */
  DRM_OUTPUT_MONO     = 1,   /*!< Always downmix the stream channel configuration to one mono channel if necessary.      */
  DRM_OUTPUT_STEREO   = 2,   /*!< Always output a two channel signal. Decoder does up-/downmixing accordingly.           */
  DRM_OUTPUT_BINAURAL = 3,   /*!< Enable binaural rendering of MPEG Surround content for headphone playback. Available
                                  only for decoders with MPEG Surround and binaural rendering support.                   */
  DRM_OUTPUT_6CHANNEL = 4    /*!< Decoder always produces a six channel output signal. If MPEG Surround is not supported,
                                  not enabled or no payload is available the decoder creates empty channels.             */
} DRMDEC_OUTPUT_MODE;

/*!
 * This enumeration defines all parameters of the DRM decoder which can be adjusted by the user at runtime.
 */
typedef enum
{
  DRM_PCM_OUTPUT_INTERLEAVED                  = 0x0000,  /*!< Define how the PCM audio output channel samples will be
                                                              addressed in the output buffer. Allowed values are
                                                              1 (interleaved) and 0 (non-interleaved audio output). */
  DRM_PCM_OUTPUT_MODE                         = 0x0001,  /*!< Set the output rendering mode. See DRMDEC_OUTPUT_MODE
                                                              for details. */
  DRM_AAC_CONCEAL_FADEOUT_SLOPE               = 0x0100,  /*!< AAC error concealment: Length of the audio signals fade-
                                                              out slope (in frames) applied for multiple consecutive
                                                              corrupted audio frames. The valid values range from
                                                              0 (no fadeout slope) to 16 (integer values only). */
  DRM_AAC_CONCEAL_FADEIN_SLOPE                = 0x0101,  /*!< AAC error concealment: Analogous to the
                                                              DRM_AAC_CONCEAL_FADEOUT_SLOPE parameter but for the
                                                              fade-in slope applied to the audio output signal after
                                                              error-free frames have arrived. */
  DRM_AAC_CONCEAL_MUTE_RELEASE                = 0x0102,  /*!< AAC error concealment: The number of error-free frames
                                                              which have to arrive at the decoder (while muting) until
                                                              the signal gets fade-in again. This parameter can be used to
                                                              avoid a pumping output signal that can be generated by
                                                              consecutive sequences of fade-in and out. The valid value
                                                              range is from 0 to 16 in integer steps. */
  DRM_AAC_CONCEAL_COMFORT_NOISE_LEVEL         = 0x0103,  /*!< AAC error concealment: The level of the noise signal that
                                                              will be inserted after a fade-out due to corrupt input
                                                              data. The valid parameter values range from 0 (full
                                                              scaled noise) to 127 (no noise insertion). */
  DRM_AAC_CONCEAL_ATTENUATION_FADEOUT_VECTOR  = 0x0104,  /*!< AAC error concealment: Vector of attenuation factors
                                                              defining the fade-out slope. It has 16 attenuation factors
                                                              which will be applied to the frames within the fade-out
                                                              process. Each factor is assigned to one frame and its
                                                              value can range from 0 to 63.75dB quantized in steps of
                                                              0.25dB. Thus the valid values for one factor range from 0
                                                              to 255 in integer steps. The data type of each factor has
                                                              to be SHORT. */
  DRM_AAC_CONCEAL_ATTENUATION_FADEIN_VECTOR   = 0x0105,  /*!< AAC error concealment: Vector of attenuation factors
                                                              defining the fade-in slope. Analogous to parameter
                                                              DRM_AAC_CONCEAL_ATTENUATION_FADEOUT_VECTOR. */
  DRM_MPEGS_ENABLE                            = 0x0500,  /*!< MPEG Surround: Allow/Disable decoding of MPS content.
                                                              Available only for decoders with MPEG Surround support. */
  DRM_MPEGS_BINAURAL_PRESET                   = 0x0501,  /*!< MPEG Surround: Binaural preset: \n
                                                               1: Dry room. \n
                                                               2: Living room (default). \n
                                                               3: Cinema. \n
                                                              Available only for decoders with MPEG Surround support.  */
  DRM_MPEGS_BINAURAL_DISTANCE                 = 0x0505,  /*!< MPEG Surround Binaural: Set perceived distance parameter.
                                                              Available only for decoders with MPEG Surround and
                                                              binaural rendering support. */
  DRM_MPEGS_BINAURAL_DIALOG_CLARITY           = 0x0506,  /*!< MPEG Surround Binaural: Define dialog clarity. Available
                                                              only for decoders with MPEG Surround and binaural
                                                              rendering support. */
  DRM_MPEGS_BINAURAL_FRONT_ANGLE              = 0x0507,  /*!< MPEG Surround Binaural: Set angle of virtual front
                                                              speakers. Available only for decoders with MPEG Surround
                                                              and binaural rendering support. */
  DRM_MPEGS_BINAURAL_BACK_ANGLE               = 0x0508,  /*!< MPEG Surround Binaural: Set angle of virtual rear
                                                              speakers. Available only for decoders with MPEG Surround
                                                              and binaural rendering support. */

  DRM_QMF_LOWPOWER                            = 0x0600,  /*!< Quadrature Mirror Filter (QMF) bank processing mode. \n
                                                              -1: Use decoder internal default (default).
                                                                  Implies MPEG Surround partially complex accordingly. \n
                                                               0: Force complex number (high quality) QMF data mode. \n
                                                               1: Force real number (low power) QMF data mode if applicable. \n
                                                              For xHE-AAC this setting is ignored. It always uses the full complex number processing. */
  DRM_QMF_RESAMPLER                           = 0x0700,  /*!< Quadrature Mirror Filter (QMF) resampler. \n
                                                              0: QMF resampler off (default, use output sampling rate defined by the SDC) \n
                                                              1: QMF resampler active (resample output to 48 kHz)\n
                                                              Note that the required output buffer size will increase by the factor (48 kHz) /
                                                              (actual codec sampling rate). This is up to a factor of 5 when using the lowest
                                                              xHE-AAC sampling rate of 9.6 kHz. */
  DRM_USAC_LP                                 = 0x0800   /*!< Special non-standard conform USAC low power mode (bit field).
                                                              This mode is not recommended if not absolutely required. \n
                                                              0x01    SBR low power harmonic QMF transposer. \n
                                                              0x02    Reserved bit. \n
                                                              0x04    Reserved bit. \n
                                                              0x08    Reserved bit. \n
                                                              0x10    Reserved bit.
                                                         */
} DRMDEC_PARAM;

/*!
 * This structure provides several informations, e.g. audio parameters like the sample rate and the number of channels
 * or informations on the DRM specific error protection mechanism.
 */
typedef struct
{
  /* audio and codec informations: */
  INT  numChannels;                                   /*!< The number of audio output channels. */
  INT  sampleRate;                                    /*!< Sample rate of the PCM audio output samples (DAC sample rate). */
  INT  bitRate;                                       /*!< The audio bit rate [bit/s]. */
  INT  frameSize;                                     /*!< Transformation length of the used codec per frame [samples]. */

  /* statistics */
  INT  nrSuperFrames;                                 /*!< Counts the decoded super frames. */
  INT  sfParseErrors;                                 /*!< Is incremented whenever the disassembly of a DRM super frame contains any errors. Note: Assembled audio frame data might still be valid. */
  INT  nrErrorFreeAUs;                                /*!< Number of audio frames which have been decoded successfully. */
  INT  nrErroneousAUs;                                /*!< Number of audio frames which have been marked as erroneous. */
  INT  lastFrameOk;                                   /*!< Flag that saves the status (OK/not OK) of the last decoded frame.
                                                           In case of lastFrameOk == 0 concealment was triggered. */
  INT  isInHoldOffMode;                               /*!< Waiting for sufficient amount of data before actual decoding can start (applies only to xHE-AAC).
                                                           Note that in this mode lastFrameOk is set to 1. */


} DRMDEC_INFO;

/*! The decoder handle identifying a decoder instance. */
typedef struct DRM_DECODER_INSTANCE *DRM_DECODER;


#ifdef __cplusplus
extern "C"
{
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * *
 *                                               *
 *    I N T E R F A C E   F u n c t i o n s :    *
 *                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * */

/*!
 * \brief Open a DRM decoder instance.
 *
 * \b Description
 *
 * Open an instance of the DRM decoder. This function allocates all needed structures and initializes them.
 * The decoder instance will be in need of a valid configuration before the first decoding of audio frames.
 * Please see DRMDEC_FHG_Config() for details.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_Open (
  DRM_DECODER        *pSelf               /*!< Pointer to a buffer to place a handle that identifies an opened DRM decoder instance after successful initialization. */
);

/*!
 * \brief Close a DRM decoder instance and free allocated memory.
 *
 * \b Description
 *
 * Close an open instance of the DRM decoder. This function deallocates all DRM decoder structures.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_Close (
  DRM_DECODER        *pSelf               /*!< Pointer to a buffer holding a handle of an open DRM decoder instance. If the function succeeds,
                                               the handle is no longer valid after this call and the buffer will be set to NULL. */
);


/*!
 * \brief Configure a decoder instance with the out-of-band audio configuration data field from the Side Description Channel (SDC).
 *
 * \b Description
 *
 * This function is used to signal the active configuration for any future DRM audio super frames.
 * Preliminary checks will determine validity but actual reconfiguration will take place with next consumed super frame.
 *
 * \note
 * - Actual configuration of internal decoder information takes place in DRMDEC_FHG_DecodeFrame() when a new DRM audio super frame is processed.
 * - Reconfigurations can be done any time and will have no effect on previously internally buffered audio frames.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_Config(
  const DRM_DECODER   self,               /*!< Handle that identifies an opened DRM decoder instance. */
  const UCHAR        *pConfig,            /*!< Pointer to buffer holding raw audio configuration (SDC data entity type 9). */
  const UINT          configLen,          /*!< Length of the configuration in bytes. */
  const UINT          asfLen,             /*!< The audio super frame length in bytes needed to decompose the audio frames correctly. */
  const UINT          hpLen,              /*!< This parameter specifies the length of the higher protected part within an audio super frame. It can be half of \b asfLen at maximum. */
  const UINT          flags               /*!< Flags field for additional special values, see DRM_CONFIG_X definitions. (e.g. DRM_CONFIG_DRM_PLUS for DRM+ mode) */
);

/*
 * Configuration flags for the DRMDEC_FHG_Config() function.
 */
#define DRM_CONFIG_NONE                       ( 0 )      /*!< Flag for no configuration options. */
#define DRM_CONFIG_DRM_PLUS                   ( 1 << 0 ) /*!< Flag telling that robustness mode E (DRM+) is used. */
#define DRM_CONFIG_FULL_RESET                 ( 1 << 1 ) /*!< Flag that triggers a full reset of the decoder instance. Internally buffered AUs are discarded and parameters set by DRMDEC_FHG_SetParam() are deleted. */
#define DRM_CONFIG_WITHOUT_ID_BITS            ( 1 << 2 ) /*!< Flag to signal that the config buffer does not include the initial 4 bit for short and stream Id. */

/*!
 * \brief Set one single decoder parameter for one single decoder instance to control its behavior at runtime.
 *
 * \remarks
 * This function returns an error if the specific parameter couldn't be set. Possible reasons could be
 * that the parameter value exceeded the valid range or that the parameter doesn't exist e.g. because
 * the decoder was built without the module to which the parameter belongs to.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_SetParam (
  const DRM_DECODER   self,               /*!< Handle that identifies an opened DRM decoder instance. */
  const DRMDEC_PARAM  decParam,           /*!< Defines the specific parameter to be set. See the definition of DRM_PARAM for a complete list
                                               of adjustable parameters and their allowed range of values. */
  const INT           value               /*!< The value which will be applied to the specific parameter. Please see the
                                               definition of DRM_PARAM for the allowed value range of the specific parameter. */
);

/*!
 * \brief Set one parameter vector for one decoder instance to control its behavior at runtime.
 *
 * \b Description
 *
 * Set a whole decoder parameter vector. To set a single parameter use DRMDEC_FHG_SetParam() instead.
 * See the definition of DRM_PARAM for a complete list of adjustable parameters and their allowed range of values.
 *
 * \remarks
 * This function returns an error if the specific parameter couldn't be set. Possible reasons could be
 * that at least one parameter value exceed the valid range or the parameter doesn't exist e.g. because
 * the decoder was built without the module to which the parameter belongs to.
*/
DRM_DECODER_ERROR
DRMDEC_FHG_SetParamVector (
  const DRM_DECODER   self,               /*!< Handle that identifies an opened DRM decoder instance. */
  const DRMDEC_PARAM  decParam,           /*!< Defines the specific parameter to be set. See the definition of DRM_PARAM for a complete list
                                               of adjustable parameters and their allowed range of values. */
  const void         *pValue              /*!< Pointer to an one dimensional parameter array. For the data type, the length and the value
                                               range of the specific parameter vector (given by param) please refer to the type definition of
                                               DRM_PARAM. */
);

/*!
 * \brief Fill decoder with DRM audio super frame data and decode one audio frame.
 *
 * \b Description
 *
 * This function is called in order to process DRM audio super frames. The input data is stored internally once a DRM audio super frame is accepted.
 * inBytesValid is decremented by the size of one audio super frame signaling that the input buffer can be released and reused.
 * Each call of this function produces PCM output samples as long as internal buffer contains valid audio frames. New audio super frames can only be accepted once all
 * internally stored audio frames have been processed. The framework is required to have the next audio super frame data ready when internal buffer runs empty.
 * Any pending configuration changes will be considered before parsing new super frame data.
 *
 * <b>Loss of super frames</b>
 *
 * The framework should signal any lost super frames by providing a dummy (zeroed out) audio super frame with the correct size in bytes.
 * The decoder will try to conceal within reasonable parameters and can fallback to internal resynchronization mode.
 *
 * \note
 * - inBytesValid must be the size of valid bytes in pInput. The output value of inBytesValid will be lowered for
 *   the amount of consumed bytes (usually the whole DRM audio super frame).
 * - The error code DRMDEC_NEED_DATA will be returned when there is no more data available in internal buffer and new audio super frame data is required.
 * - Framework is permitted to use the DRM_DECODE_CONCEAL_AUDIO_FRAME flag if no further audio super frames are available in time and output samples are needed.
 *
 * \remarks
 * The output data delivered by this function consists of 16 bit linear PCM sample values in native host byte order.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_DecodeFrame(
  DRM_DECODER         self,               /*!< Handle of an open DRM decoder instance. */
  const UCHAR        *pInput,             /*!< Address of a buffer holding the DRM super frames. The buffer has to have at least the size of one audio super
                                               frame. This function expects the beginning of the next DRM super frame to read at the provided address. */
  UINT               *inBytesValid,       /*!< Address of a buffer holding the fill level of the input buffer. If the function reads a DRM super frame
                                               from the input buffer this value is going to be decreased at function return. When filling a new DRM
                                               audio super frame into the input buffer the value has to be increased by the framework. */
  INT_PCM            *pSamples,           /*!< Address of a buffer receiving the decoded PCM output data. */
  INT                *nSamples,           /*!< Address of an INT receiving the number of samples written to buffer pointed to by \b pSamples. A value
                                               greater than 0 indicates than one audio frame has been decoded. */
  INT                 flags,              /*!< Flags to requests conditional processing of follow up data, see DRM_DECODE_X definitions. */
  DRMDEC_INFO       **pDecoderInfo        /*!< Address of a pointer receiving the address pointing on the internal DRMDEC_INFO structure. This structure
                                               contains the necessary information for target audio output format (e.g.: number of channels, sampling frequency, frame size) */
);

/*
 * Decoding flags for the DRMDEC_FHG_DecodeFrame() function.
 */
#define DRM_DECODE_NONE                       ( 0 )      /*!< Flag for no decoding options. */
#define DRM_DECODE_FORCE_OUTPUT               ( 1 << 0 ) /*!< This flag can be used when requesting output samples.<br>
                                                              It tells the decoder that the framework requests continuous output samples even if no more valid received audio data is
                                                              available for processing. The provided output samples include fading, with silence in between.<br>
                                                              The flag is ignored as long as valid received audio data is internally available, and known audio frames with invalid
                                                              content are always concealed (irrespective of this flag).<br>
                                                              This flag is only considered if audio input data is missing, i.e. whenever the output function would otherwise provide
                                                              DRMDEC_NEED_DATA instead of output samples. */
#define DRM_DECODE_SUPERFRAME_CORRUPT         ( 1 << 1 ) /*!< This flag is deprecated and might be removed in future versions. The preferred way to signal a lost or heavily corrupted
                                                              super frame is by providing a zeroed out dummy super frame of correct size in bytes.<br>
                                                              The flag tells the decoder that the the provided audio super frame data is corrupt (e.g. was not received or does not contain
                                                              any useful content).
                                                              If multiple consecutive audio super frames are missing, the flag shall be provided once for each
                                                              missing audio super frame. Once an unknown number of audio super frames was missing (e.g. due to complete interruption of
                                                              reception), the DRM_DECODE_RESYNC_REQUIRED flag shall be provided in addition to this flag. */
#define DRM_DECODE_RESYNC_REQUIRED            ( 1 << 2 ) /*!< This flag can be used when providing an audio super frame.<br>
                                                              It tells the decoder that an unknown number of input data is missing (e.g. one or more audio super frames).
                                                              If provided along with the next valid audio super frame data, the resync is performed before decoding the new audio super
                                                              frame.<br>
                                                              After providing this flag once, no further audio super frame dummies need to be provided. This flag is a last resort.
                                                              Under normal conditions a full sequence of dummy audio super frames is preferred as might enable the decoder to determine
                                                              the number of missed audio super frames. */

/*!
 * \brief Get the I/O buffer minimum requirements.
 */
DRM_DECODER_ERROR
DRMDEC_FHG_GetIoBufInfo (
  CDK_bufDescr       *pBufDescr           /*!< Pointer to a pre-initialized buffer description structure with at least DRMDEC_MAX_NUM_IO_BUFFERS entries. */
);

/*!
 * \brief Get decoder library info.
 */
INT
DRMDEC_FHG_GetLibInfo(
  LIB_INFO           *info                /*!< Pointer to a client provided LIB_INFO structure that will be filled with the decoder library information. */
);

/*!
 * \brief Get memory required to set up one decoder instance.
 */
void DRMDEC_FHG_GetRequiredMem( CDK_MEM_TAB *pMemTab );


#ifdef __cplusplus
}
#endif

#endif /* DRM_DECODER_LIB_H */
